<?php
// all-pages.php
session_start();
require_once "../config/db.php"; // loads $pdo (PDO connection)

// --- Pagination setup ---
$limit = 100;
$page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int) $_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// --- Search filter ---
$search = "";
$whereClause = "";
$params = [];

if (!empty($_GET['search'])) {
    $search = trim($_GET['search']);
    $whereClause = "WHERE title LIKE :search OR slug LIKE :search";
    $params[':search'] = "%" . $search . "%";
}

// --- Count total pages ---
$countSql = "SELECT COUNT(*) FROM pages $whereClause";
$countStmt = $pdo->prepare($countSql);
$countStmt->execute($params);
$totalRows = $countStmt->fetchColumn();
$totalPages = ceil($totalRows / $limit);

// --- Fetch paginated results ---
$sql = "SELECT id, title, slug, created_at 
        FROM pages 
        $whereClause 
        ORDER BY created_at DESC 
        LIMIT :limit OFFSET :offset";
$stmt = $pdo->prepare($sql);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value, PDO::PARAM_STR);
}
$stmt->bindValue(":limit", $limit, PDO::PARAM_INT);
$stmt->bindValue(":offset", $offset, PDO::PARAM_INT);
$stmt->execute();
$pages = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>All Generated Pages</title>
    <link rel="stylesheet" href="./assets/css/bootstrap.min.css">
    <link rel="stylesheet" href="./assets/css/slick.css">
    <link rel="stylesheet" href="./assets/css/style.css">
    <link rel="stylesheet" href="./assets/css/responsive.css">
</head>
<body>
    <div class="body-bg">
        <?php include 'includes/sidebar.php'; ?>

        <section class="nftmax-adashboard nftmax-show">
        <h2 class="mb-4">All Generated Pages</h2>

<!-- 🔎 Search -->
<form method="get" class="row g-2 mb-4">
    <div class="col-md-4">
        <input type="text" name="search" value="<?= htmlspecialchars($search) ?>" class="form-control" placeholder="Search by title or slug...">
    </div>
    <div class="col-md-2">
        <button type="submit" class="btn btn-primary w-100">Search</button>
    </div>
    <?php if ($search): ?>
    <div class="col-md-2">
        <a href="all-pages.php" class="btn btn-secondary w-100">Reset</a>
    </div>
    <?php endif; ?>
</form>

<?php echo "<pre>Total Rows: $totalRows | Limit: $limit | Total Pages: $totalPages | Current Page: $page</pre>";
?>
<?php if (!empty($pages)): ?>
    <div class="table-responsive">
        <table class="table table-bordered table-hover align-middle">
            <thead class="table-dark">
                <tr>
                    <th>ID</th>
                    <th>Title</th>
                    <th>Slug</th>
                    <th>Created At</th>
                    <th>View</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($pages as $row): ?>
                    <tr>
                        <td><?= $row["id"] ?></td>
                        <td><?= htmlspecialchars($row["title"]) ?></td>
                        <td><?= htmlspecialchars($row["slug"]) ?></td>
                        <td><?= $row["created_at"] ?></td>
                        <td>
                            <a target="_blank" class="btn btn-sm btn-secondary"
                               href="/page.php?slug=<?= urlencode($row["slug"]) ?>">
                                View
                            </a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <nav>
<ul class="pagination justify-content-center">

<!-- Previous -->
<li class="page-item <?= ($page <= 1) ? 'disabled' : '' ?>">
  <a class="page-link" href="?page=<?= max(1, $page-1) ?>&search=<?= urlencode(is_string($search) ? $search : '') ?>">Previous</a>
</li>

<?php
$start = max(1, $page - 2);
$end   = min($totalPages, $page + 2);

// show first page + ellipsis if needed
if ($start > 1) {
    echo '<li class="page-item"><a class="page-link" href="?page=1&search='.urlencode(is_string($search) ? $search : '').'">1</a></li>';
    if ($start > 2) {
        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
    }
}

// main loop
for ($i = $start; $i <= $end; $i++) {
    $active = ($i == $page) ? 'active' : '';
    echo '<li class="page-item '.$active.'"><a class="page-link" href="?page='.$i.'&search='.urlencode(is_string($search) ? $search : '').'">'.$i.'</a></li>';
}

// show last page + ellipsis if needed
if ($end < $totalPages) {
    if ($end < $totalPages - 1) {
        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
    }
    echo '<li class="page-item"><a class="page-link" href="?page='.$totalPages.'&search='.urlencode(is_string($search) ? $search : '').'">'.$totalPages.'</a></li>';
}
?>

<!-- Next -->
<li class="page-item <?= ($page >= $totalPages) ? 'disabled' : '' ?>">
  <a class="page-link" href="?page=<?= min($totalPages, $page+1) ?>&search=<?= urlencode(is_string($search) ? $search : '') ?>">Next</a>
</li>

</ul>
</nav>




<?php else: ?>
    <div class="alert alert-warning">No pages found.</div>
<?php endif; ?>
        </section>
    </div>
</body>
</html>
